﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Diagnostics;
using RevisionAnalyser.Global;
using System.IO;

namespace RecoderProject
{
    public delegate void StatusChangedEventHandler(object sender, StatusChangedEventArgs e);
    public delegate void ErrorReceivedEventHandler(object sender, ErrorReceivedEventArgs e);
    public delegate void WarningReceivedEventHandler(object sender, WarningReceivedEventArgs e);
    public delegate void ProcessCompleteEventHandler(object sender, ProcessCompleteEventArgs e);

    public class StatusChangedEventArgs : System.EventArgs
    {
        public StatusChangedEventArgs(string message, int value, StatusType type)
        {
            Message = message;
            Value = value;
            Type = type;
        }

        public string Message
        {
            get;
            private set;
        }

        public int Value
        {
            get;
            private set;
        }

        public StatusType Type
        {
            get;
            private set;
        }
    }
    public class ErrorReceivedEventArgs : System.EventArgs
    {
        public ErrorReceivedEventArgs(string message, int number)
        {
            Message = message;
            Number = number;
        }

        public string Message
        {
            get;
            private set;
        }

        public int Number
        {
            get;
            private set;
        }
    }
    public class WarningReceivedEventArgs : System.EventArgs
    {
        public WarningReceivedEventArgs(string message)
        {
            Message = message;
        }

        public string Message
        {
            get;
            private set;
        }
    }
    public class ProcessCompleteEventArgs : System.EventArgs
    {
        public ProcessCompleteEventArgs(string message)
        {
            Message = message;
        }

        public string Message
        {
            get;
            private set;
        }
    }

    public enum StatusType { MESSAGE, VALUE };
    
    public class RecoderAnalyser
    {
        private static string PROGRAM_PATH = String.Format("-jar \"{0}\\RecoderAnalyser.jar\"", Constants.PLUGINS_PATH);

        public event StatusChangedEventHandler StatusChanged;
        public event ErrorReceivedEventHandler ErrorReceived;
        public event WarningReceivedEventHandler WarningReceived;
        public event ProcessCompleteEventHandler ProcessComplete;

        private Process _process;
        private Boolean result = false;

        public RecoderAnalyser(string javaPath)
        {
            _process = new Process();
            _process.StartInfo.FileName = String.Format("\"{0}java.exe\"", javaPath);
            _process.StartInfo.RedirectStandardOutput = true;
            _process.StartInfo.RedirectStandardError = true;
            _process.StartInfo.UseShellExecute = false;
            _process.StartInfo.CreateNoWindow = true;
            _process.ErrorDataReceived += new DataReceivedEventHandler(ErrorDataReceived);
            _process.OutputDataReceived += new DataReceivedEventHandler(OutputDataReceived);
        }

        public bool AnalyseRevision(string sourceFolder, string outputPath, long revision, List<string> libs, Boolean ignore)
        {
            StringBuilder args = new StringBuilder(String.Format("{0} -analyse \"{1}\" -revision {2} -output \"{3}\"", PROGRAM_PATH, sourceFolder, revision, outputPath));
            args.Append(libsToArguments(libs));

            //Check if ignore errors is set
            if (ignore)
            {
                args.Append(" -ignore");
            }

            //Check if there is already a file to merge
            if (File.Exists(outputPath))
            {
                args.Append(String.Format(" -merge \"{0}\"", outputPath));
            }

            //Run with the generated arguments
            Run(args.ToString());
            return result;
        }

        public void RemoveRevision(string inputPath, long revision)
        {
            Run(String.Format("{0} -input \"{1}\" -remove {2}", PROGRAM_PATH, inputPath, revision)); 
        }

        public void CreateSolidSXDatabase(string inputPath, string dbPath)
        {
            Run(String.Format("{0} -input \"{1}\" -solidsx2db \"{2}\"", PROGRAM_PATH, inputPath, dbPath));
        }

        private void Run(string args)
        {
            _process.StartInfo.Arguments = args;
            _process.Start();
            _process.BeginErrorReadLine();
            _process.BeginOutputReadLine();
            _process.WaitForExit();
        }

        private string libsToArguments(List<string> libs)
        {
            StringBuilder files = new StringBuilder();
            StringBuilder folders = new StringBuilder();

            foreach (string lib in libs)
            {
                if (File.Exists(lib))
                {
                    if (files.ToString().Equals(String.Empty))
                    {
                        files.Append(" -libfiles \"");
                    }
                    files.Append(lib).Append(";");
                }
                else
                {
                    if (folders.ToString().Equals(String.Empty))
                    {
                        folders.Append(" -libfolders \"");
                    }
                    folders.Append(lib).Append(";");
                }
            }

            if (!files.ToString().Equals(String.Empty))
            {
                files.Remove(files.Length - 1, 1);
                files.Append("\"");
            }

            if (!folders.ToString().Equals(String.Empty))
            {
                folders.Remove(folders.Length - 1, 1);
                folders.Append("\"");
            }
            
            return files.Append(folders).ToString();
        }

        private void OutputDataReceived(object sender, DataReceivedEventArgs e)
        {
            if (e.Data != null)
            {
                if (e.Data.StartsWith("STATUS_VALUE:"))
                {
                    int statusValue = int.Parse(e.Data.Substring(13, e.Data.IndexOf('%') - 13));
                    this.StatusChanged(this, new StatusChangedEventArgs(null, statusValue, StatusType.VALUE));
                }
                else if (e.Data.StartsWith("STATUS_MESSAGE:"))
                {
                    string statusMessage = e.Data.Substring(e.Data.IndexOf(':') + 2);
                    this.StatusChanged(this, new StatusChangedEventArgs(statusMessage, 0, StatusType.MESSAGE));
                }

                if (e.Data.StartsWith("STATUS_EXIT"))
                {
                    result = true;
                    Exit("Finished without errors");
                }
            }
        }

        private void ErrorDataReceived(object sender, DataReceivedEventArgs e)
        {
            if (e.Data != null)
            {
                Console.WriteLine("Error: " + e.Data);
                if (e.Data.StartsWith("ERROR_NUM"))
                {
                    int errorNumber = int.Parse(e.Data.Substring(9, e.Data.IndexOf(':') - 9));
                    string errorMessage = e.Data.Substring(e.Data.IndexOf(':') + 2);
                    this.ErrorReceived(this, new ErrorReceivedEventArgs(errorMessage, errorNumber));
                }
                else if (e.Data.StartsWith("WARNING"))
                {
                    string warningMessage = e.Data.Substring(e.Data.IndexOf(':') + 2);
                    this.WarningReceived(this, new WarningReceivedEventArgs(warningMessage));
                }
                else if (e.Data.StartsWith("ERROR_EXIT"))
                {
                    Exit("Finished with errors");
                }
            }
        }

        private void Exit(string message)
        {
            _process.CancelErrorRead();
            _process.CancelOutputRead();
            this.ProcessComplete(this, new ProcessCompleteEventArgs(message));
        }
    }
}
